package org.dromara.zjk.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import lombok.RequiredArgsConstructor;
import org.dromara.common.core.utils.MapstructUtils;
import org.dromara.common.core.utils.StringUtils;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.zjk.domain.ZjkExpertMajor;
import org.dromara.zjk.domain.bo.ZjkExpertMajorBo;
import org.dromara.zjk.domain.vo.ZjkExpertMajorVo;
import org.dromara.zjk.mapper.ZjkExpertMajorMapper;
import org.dromara.zjk.service.IZjkExpertMajorService;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * 专家库 专业关系表Service业务层处理
 *
 * @author ygh
 * @date 2024-11-01
 */
@RequiredArgsConstructor
@Service
public class ZjkExpertMajorServiceImpl implements IZjkExpertMajorService {

    private final ZjkExpertMajorMapper baseMapper;

    @Override
    public void saveBatch(List<ZjkExpertMajor> zjkExpertMajorList) {
        baseMapper.insertBatch(zjkExpertMajorList);
    }

    /**
     * 查询专家库 专业关系表
     *
     * @param id 主键
     * @return 专家库 专业关系表
     */
    @Override
    public ZjkExpertMajorVo queryById(Long id){
        return baseMapper.selectVoById(id);
    }

    /**
     * 分页查询专家库 专业关系表列表
     *
     * @param bo        查询条件
     * @param pageQuery 分页参数
     * @return 专家库 专业关系表分页列表
     */
    @Override
    public TableDataInfo<ZjkExpertMajorVo> queryPageList(ZjkExpertMajorBo bo, PageQuery pageQuery) {
        LambdaQueryWrapper<ZjkExpertMajor> lqw = buildQueryWrapper(bo);
        Page<ZjkExpertMajorVo> result = baseMapper.selectVoPage(pageQuery.build(), lqw);
        return TableDataInfo.build(result);
    }

    /**
     * 查询符合条件的专家库 专业关系表列表
     *
     * @param bo 查询条件
     * @return 专家库 专业关系表列表
     */
    @Override
    public List<ZjkExpertMajorVo> queryList(ZjkExpertMajorBo bo) {
        LambdaQueryWrapper<ZjkExpertMajor> lqw = buildQueryWrapper(bo);
        return baseMapper.selectVoList(lqw);
    }

    private LambdaQueryWrapper<ZjkExpertMajor> buildQueryWrapper(ZjkExpertMajorBo bo) {
        Map<String, Object> params = bo.getParams();
        LambdaQueryWrapper<ZjkExpertMajor> lqw = Wrappers.lambdaQuery();
        lqw.eq(bo.getUserId() != null, ZjkExpertMajor::getUserId, bo.getUserId());
        lqw.eq(bo.getExpertId() != null, ZjkExpertMajor::getExpertId, bo.getExpertId());
        lqw.eq(StringUtils.isNotBlank(bo.getMajor()), ZjkExpertMajor::getMajor, bo.getMajor());
        lqw.eq(bo.getDegree() != null, ZjkExpertMajor::getDegree, bo.getDegree());
        lqw.eq(bo.getStatus() != null, ZjkExpertMajor::getStatus, bo.getStatus());
        return lqw;
    }

    /**
     * 新增专家库 专业关系表
     *
     * @param bo 专家库 专业关系表
     * @return 是否新增成功
     */
    @Override
    public Boolean insertByBo(ZjkExpertMajorBo bo) {
        ZjkExpertMajor add = MapstructUtils.convert(bo, ZjkExpertMajor.class);
        validEntityBeforeSave(add);
        boolean flag = baseMapper.insert(add) > 0;
        if (flag) {
            bo.setId(add.getId());
        }
        return flag;
    }

    /**
     * 修改专家库 专业关系表
     *
     * @param bo 专家库 专业关系表
     * @return 是否修改成功
     */
    @Override
    public Boolean updateByBo(ZjkExpertMajorBo bo) {
        ZjkExpertMajor update = MapstructUtils.convert(bo, ZjkExpertMajor.class);
        validEntityBeforeSave(update);
        return baseMapper.updateById(update) > 0;
    }

    /**
     * 保存前的数据校验
     */
    private void validEntityBeforeSave(ZjkExpertMajor entity){
        //TODO 做一些数据校验,如唯一约束
    }

    /**
     * 校验并批量删除专家库 专业关系表信息
     *
     * @param ids     待删除的主键集合
     * @param isValid 是否进行有效性校验
     * @return 是否删除成功
     */
    @Override
    public Boolean deleteWithValidByIds(Collection<Long> ids, Boolean isValid) {
        if(isValid){
            //TODO 做一些业务上的校验,判断是否需要校验
        }
        return baseMapper.deleteByIds(ids) > 0;
    }

    /**
     * 根据用户 id  专家 id 进行 新增专业学历信息
     * @param majorList
     * @param expertId
     * @param userId
     */
    @Override
    public void handleExpertMajors(List<ZjkExpertMajor> majorList, Long expertId, Long userId) {
        // 删除旧的专业学历信息
        baseMapper.delete(new LambdaQueryWrapper<ZjkExpertMajor>()
            .eq(ZjkExpertMajor::getExpertId, expertId)
            .eq(ZjkExpertMajor::getUserId, userId)
        );
        if (majorList != null &&!majorList.isEmpty()){
            // 设置 userId 和 expertId 并批量插入新数据
            majorList.forEach(major -> {
                major.setUserId(userId);
                major.setExpertId(expertId);
            });
            baseMapper.insertOrUpdateBatch(majorList);
        }
    }

    /**
     * 根据用户 id 专家 id 获取专业关系
     * @param userId
     * @param expertId
     * @return
     */
    @Override
    public List<ZjkExpertMajor> populateExpertMajorList(Long userId, Long expertId) {
        List<ZjkExpertMajor> majorList = baseMapper.selectList(
            new LambdaQueryWrapper<ZjkExpertMajor>()
                .eq(ZjkExpertMajor::getExpertId, expertId)
                .eq(ZjkExpertMajor::getUserId, userId)
        );
        return majorList.isEmpty() ? new ArrayList<>() : majorList;
    }

    @Override
    public List<Map<Integer,Integer>>  getExpertMajorCount() {
        return baseMapper.getExpertMajorCount();
    }
}
