package org.dromara.zjk.service.impl;

import org.dromara.common.core.utils.MapstructUtils;
import org.dromara.common.core.utils.StringUtils;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.mybatis.core.page.PageQuery;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import lombok.RequiredArgsConstructor;
import org.dromara.common.satoken.utils.LoginHelper;
import org.dromara.zjk.domain.ZjkProductSummarize;
import org.dromara.zjk.domain.ZjkProductSummarizeLog;
import org.dromara.zjk.domain.ZjkProjectEvaluation;
import org.dromara.zjk.domain.bo.ZjkProjectEvaluationBo;
import org.dromara.zjk.domain.vo.ZjkProjectEvaluationVo;
import org.dromara.zjk.mapper.ZjkProductSummarizeLogMapper;
import org.dromara.zjk.mapper.ZjkProductSummarizeMapper;
import org.dromara.zjk.service.IZjkProjectEvaluationService;
import org.springframework.stereotype.Service;
import org.dromara.zjk.mapper.ZjkProjectEvaluationMapper;

import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Collection;

/**
 * 项目总结评价Service业务层处理
 *
 * @author ygh
 * @date 2024-11-08
 */
@RequiredArgsConstructor
@Service
public class ZjkProjectEvaluationServiceImpl implements IZjkProjectEvaluationService {

    private final ZjkProjectEvaluationMapper baseMapper;

    private final ZjkProductSummarizeLogMapper zjkProductSummarizeLogMapper;

    /**
     * 查询项目总结评价
     *
     * @param id 主键
     * @return 项目总结评价
     */
    @Override
    public ZjkProjectEvaluationVo queryById(Long id){
        return baseMapper.selectVoById(id);
    }

    /**
     * 分页查询项目总结评价列表
     *
     * @param bo        查询条件
     * @param pageQuery 分页参数
     * @return 项目总结评价分页列表
     */
    @Override
    public TableDataInfo<ZjkProjectEvaluationVo> queryPageList(ZjkProjectEvaluationBo bo, PageQuery pageQuery) {
        LambdaQueryWrapper<ZjkProjectEvaluation> lqw = buildQueryWrapper(bo);
        Page<ZjkProjectEvaluationVo> result = baseMapper.selectVoPage(pageQuery.build(), lqw);
        return TableDataInfo.build(result);
    }

    /**
     * 查询符合条件的项目总结评价列表
     *
     * @param bo 查询条件
     * @return 项目总结评价列表
     */
    @Override
    public List<ZjkProjectEvaluationVo> queryList(ZjkProjectEvaluationBo bo) {
        LambdaQueryWrapper<ZjkProjectEvaluation> lqw = buildQueryWrapper(bo);
        return baseMapper.selectVoList(lqw);
    }

    private LambdaQueryWrapper<ZjkProjectEvaluation> buildQueryWrapper(ZjkProjectEvaluationBo bo) {
        Map<String, Object> params = bo.getParams();
        LambdaQueryWrapper<ZjkProjectEvaluation> lqw = Wrappers.lambdaQuery();
        lqw.eq(bo.getProductSummarizeId() != null, ZjkProjectEvaluation::getProductSummarizeId, bo.getProductSummarizeId());
        lqw.eq(bo.getGoalAchievement() != null, ZjkProjectEvaluation::getGoalAchievement, bo.getGoalAchievement());
        lqw.eq(bo.getDeliveryQuality() != null, ZjkProjectEvaluation::getDeliveryQuality, bo.getDeliveryQuality());
        lqw.eq(bo.getCostControl() != null, ZjkProjectEvaluation::getCostControl, bo.getCostControl());
        lqw.eq(bo.getRiskManagement() != null, ZjkProjectEvaluation::getRiskManagement, bo.getRiskManagement());
        lqw.eq(bo.getProjectValue() != null, ZjkProjectEvaluation::getProjectValue, bo.getProjectValue());
        lqw.eq(StringUtils.isNotBlank(bo.getEvaluateFile()), ZjkProjectEvaluation::getEvaluateFile, bo.getEvaluateFile());
        lqw.eq(StringUtils.isNotBlank(bo.getProjectSummaryEvaluation()), ZjkProjectEvaluation::getProjectSummaryEvaluation, bo.getProjectSummaryEvaluation());
        return lqw;
    }

    /**
     * 新增项目总结评价
     *
     * @param bo 项目总结评价
     * @return 是否新增成功
     */
    @Override
    public Boolean insertByBo(ZjkProjectEvaluationBo bo) {
        //修改元数据为已评价
        ZjkProductSummarizeLog zjkProductSummarize = new ZjkProductSummarizeLog();
        zjkProductSummarize.setId(bo.getProductSummarizeId());
        zjkProductSummarize.setIsEvaluate(1);
        zjkProductSummarizeLogMapper.updateById(zjkProductSummarize);
        ZjkProjectEvaluation add = MapstructUtils.convert(bo, ZjkProjectEvaluation.class);
        add.setAppraiser(LoginHelper.getUsername());
        add.setAppraiserTime(new Date());
        validEntityBeforeSave(add);
        boolean flag = baseMapper.insert(add) > 0;
        if (flag) {
            bo.setId(add.getId());
        }
        return flag;
    }

    /**
     * 修改项目总结评价
     *
     * @param bo 项目总结评价
     * @return 是否修改成功
     */
    @Override
    public Boolean updateByBo(ZjkProjectEvaluationBo bo) {
        ZjkProjectEvaluation update = MapstructUtils.convert(bo, ZjkProjectEvaluation.class);
        validEntityBeforeSave(update);
        return baseMapper.updateById(update) > 0;
    }

    /**
     * 保存前的数据校验
     */
    private void validEntityBeforeSave(ZjkProjectEvaluation entity){
        //TODO 做一些数据校验,如唯一约束
    }

    /**
     * 校验并批量删除项目总结评价信息
     *
     * @param ids     待删除的主键集合
     * @param isValid 是否进行有效性校验
     * @return 是否删除成功
     */
    @Override
    public Boolean deleteWithValidByIds(Collection<Long> ids, Boolean isValid) {
        if(isValid){
            //TODO 做一些业务上的校验,判断是否需要校验
        }
        return baseMapper.deleteByIds(ids) > 0;
    }
}
