package org.dromara.zjk.service.impl;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.collection.CollectionUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import lombok.RequiredArgsConstructor;
import org.dromara.common.core.constant.TenantConstants;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.exception.ServiceException;
import org.dromara.common.core.utils.MapstructUtils;
import org.dromara.common.core.utils.StringUtils;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.satoken.utils.LoginHelper;
import org.dromara.zjk.domain.ZjkReviewPhase;
import org.dromara.zjk.domain.bo.ZjkProductBo;
import org.dromara.zjk.domain.bo.ZjkReviewPhaseBo;
import org.dromara.zjk.domain.bo.ZjkReviewProductBo;
import org.dromara.zjk.domain.vo.ZjkExpertVo;
import org.dromara.zjk.domain.vo.ZjkProductVo;
import org.dromara.zjk.domain.vo.ZjkReviewPhaseVo;
import org.dromara.zjk.domain.vo.ZjkReviewProductVo;
import org.dromara.zjk.enums.ReviewSnEnum;
import org.dromara.zjk.mapper.ZjkExpertMapper;
import org.dromara.zjk.mapper.ZjkProductMapper;
import org.dromara.zjk.mapper.ZjkReviewPhaseMapper;
import org.dromara.zjk.service.IZjkReviewPhaseService;
import org.dromara.zjk.zwy.utils.StringCryptoUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 *抽取条件Service业务层处理
 *
 * @author ygh
 * @date 2024-11-01
 */
@RequiredArgsConstructor
@Service
public class ZjkReviewPhaseServiceImpl implements IZjkReviewPhaseService {

    private final ZjkReviewPhaseMapper baseMapper;

    private final ZjkProductMapper productMapper;
    private final ZjkExpertMapper expertMapper;
    @Autowired
    private StringCryptoUtil stringCryptoUtil;
    /**
     * 查询抽取条件
     *
     * @param tenantId 主键
     * @return 抽取条件
     */
    @Override
    public ZjkReviewPhaseVo queryById(String tenantId){
        return baseMapper.selectVoById(tenantId);
    }

    /**
     * 分页查询抽取条件列表
     *
     * @param bo        查询条件
     * @param pageQuery 分页参数
     * @return 抽取条件分页列表
     */
    @Override
    public TableDataInfo<ZjkReviewPhaseVo> queryPageList(ZjkReviewPhaseBo bo, PageQuery pageQuery) {
        LambdaQueryWrapper<ZjkReviewPhase> lqw = buildQueryWrapper(bo);
        Page<ZjkReviewPhaseVo> result = baseMapper.selectVoPage(pageQuery.build(), lqw);
        return TableDataInfo.build(result);
    }

    /**
     * 查询符合条件的抽取条件列表
     *
     * @param bo 查询条件
     * @return 抽取条件列表
     */
    @Override
    public List<ZjkReviewPhaseVo> queryList(ZjkReviewPhaseBo bo) {
        LambdaQueryWrapper<ZjkReviewPhase> lqw = buildQueryWrapper(bo);
        return baseMapper.selectVoList(lqw);
    }

    private LambdaQueryWrapper<ZjkReviewPhase> buildQueryWrapper(ZjkReviewPhaseBo bo) {
        Map<String, Object> params = bo.getParams();
        LambdaQueryWrapper<ZjkReviewPhase> lqw = Wrappers.lambdaQuery();
        lqw.eq(StringUtils.isNotBlank(bo.getTenantId()), ZjkReviewPhase::getTenantId, bo.getTenantId());
        lqw.eq(bo.getREVISION() != null, ZjkReviewPhase::getRevision, bo.getREVISION());
        lqw.eq(StringUtils.isNotBlank(bo.getReviewSn()), ZjkReviewPhase::getReviewSn, bo.getReviewSn());
        lqw.eq(bo.getReviewTime() != null, ZjkReviewPhase::getReviewTime, bo.getReviewTime());
        lqw.eq(StringUtils.isNotBlank(bo.getReviewPlace()), ZjkReviewPhase::getReviewPlace, bo.getReviewPlace());
        lqw.eq(StringUtils.isNotBlank(bo.getPlannedIntake()), ZjkReviewPhase::getPlannedIntake, bo.getPlannedIntake());
        lqw.eq(StringUtils.isNotBlank(bo.getExtractionMode()), ZjkReviewPhase::getExtractionMode, bo.getExtractionMode());
        lqw.eq(StringUtils.isNotBlank(bo.getExtractioRestriction()), ZjkReviewPhase::getExtractioRestriction, bo.getExtractioRestriction());
        lqw.eq(StringUtils.isNotBlank(bo.getAvoidanceExpert()), ZjkReviewPhase::getAvoidanceExpert, bo.getAvoidanceExpert());
        lqw.eq(StringUtils.isNotBlank(bo.getAvoidanceUnit()), ZjkReviewPhase::getAvoidanceUnit, bo.getAvoidanceUnit());
        return lqw;
    }

    /**
     * 新增抽取条件
     *
     * @param bo 抽取条件
     * @return 是否新增成功
     */
    @Override
    public Boolean insertByBo(ZjkReviewPhaseBo bo) {
        ZjkReviewPhase add = MapstructUtils.convert(bo, ZjkReviewPhase.class);
        validEntityBeforeSave(add);
        boolean flag = baseMapper.insert(add) > 0;
        if (flag) {
            bo.setTenantId(add.getTenantId());
        }
        return flag;
    }

    /**
     * 修改抽取条件
     *
     * @param bo 抽取条件
     * @return 是否修改成功
     */
    @Override
    public Boolean updateByBo(ZjkReviewPhaseBo bo) {
        ZjkReviewPhase update = MapstructUtils.convert(bo, ZjkReviewPhase.class);
        validEntityBeforeSave(update);
        return baseMapper.updateById(update) > 0;
    }

    /**
     * 保存前的数据校验
     */
    private void validEntityBeforeSave(ZjkReviewPhase entity){
        //TODO 做一些数据校验,如唯一约束
    }

    /**
     * 校验并批量删除抽取条件信息
     *
     * @param ids     待删除的主键集合
     * @param isValid 是否进行有效性校验
     * @return 是否删除成功
     */
    @Override
    public Boolean deleteWithValidByIds(Collection<String> ids, Boolean isValid) {
        if(isValid){
            //TODO 做一些业务上的校验,判断是否需要校验
        }
        return baseMapper.deleteByIds(ids) > 0;
    }

    @Override
    public TableDataInfo<ZjkReviewProductVo> reviewProductList(ZjkReviewProductBo bo, PageQuery pageQuery) {
        boolean contains = LoginHelper.getLoginUser().getRolePermission().contains(TenantConstants.EXPERT_ROLE_KEY);
        boolean xmcontains = LoginHelper.getLoginUser().getRolePermission().contains(TenantConstants.XM_ROLE_KEY);
        boolean jgcontains = LoginHelper.getLoginUser().getRolePermission().contains(TenantConstants.JG_ROLE_KEY);

        bo.setUserId(LoginHelper.getUserId());
        ZjkExpertVo zjkExpertVo = expertMapper.selectByUserId(LoginHelper.getUserId());
        if(zjkExpertVo!=null){
            bo.setExpertId(zjkExpertVo.getExpertId());
        }

        if (contains) {
            bo.setIsExpert(1);
        }else if(xmcontains){
            bo.setIsExpert(0);
        }else{
            bo.setIsExpert(2);
        }
        //机构项目管理员
        Page<ZjkReviewProductVo> result = baseMapper.reviewProductList(pageQuery.build(), bo);
        return TableDataInfo.build(result);
    }

    @Override
    public ZjkReviewProductVo reviewListById(Long id) {
        //ZjkReviewPhaseVo zjkReviewPhaseVo = baseMapper.selectVoById(id);
        //if(zjkReviewPhaseVo == null){
        //    return new ZjkReviewProductVo();
        //}

        ZjkProductVo zjkProductVo = productMapper.selectVoById(id);
        if(zjkProductVo==null){
            throw new ServiceException("无效项目");
        }
        ZjkReviewProductVo zjkReviewProductVo = BeanUtil.copyProperties(zjkProductVo, ZjkReviewProductVo.class);

        List<Long> reviewIdList = baseMapper.getByProductId(id);

        LambdaQueryWrapper<ZjkReviewPhase> wrapper = new LambdaQueryWrapper<>();
        wrapper.in(ZjkReviewPhase::getId, reviewIdList);
        wrapper.eq(ZjkReviewPhase::getEffective, 1);
        List<ZjkReviewPhaseVo> zjkReviewPhaseVos = baseMapper.selectVoList(wrapper);
        zjkReviewProductVo.setPhaseVoList(zjkReviewPhaseVos);
        return zjkReviewProductVo;
    }

    @Override
    public List<ZjkReviewProductVo> reviewExport(ZjkReviewProductBo bo) {
        boolean contains = LoginHelper.getLoginUser().getRolePermission().contains(TenantConstants.EXPERT_ROLE_KEY);
        boolean xmcontains = LoginHelper.getLoginUser().getRolePermission().contains(TenantConstants.XM_ROLE_KEY);

        bo.setUserId(LoginHelper.getUserId());
        ZjkExpertVo zjkExpertVo = expertMapper.selectByUserId(LoginHelper.getUserId());
        if(zjkExpertVo!=null){
            bo.setExpertId(zjkExpertVo.getExpertId());
        }

        if (contains) {
            bo.setIsExpert(1);
        }else if(xmcontains){
            bo.setIsExpert(0);
        }else{
            bo.setIsExpert(2);
        }
        List<ZjkReviewProductVo> result = baseMapper.reviewProductListForExport(bo);
        if(!CollectionUtil.isEmpty(result)){
            for (ZjkReviewProductVo zjkReviewProductVo : result) {
                zjkReviewProductVo.setUnitConcat(stringCryptoUtil.decryptField(zjkReviewProductVo.getUnitConcat()));
                String reviewSn = zjkReviewProductVo.getReviewSn();
                ReviewSnEnum byCode = ReviewSnEnum.getByCode(reviewSn);
                if(byCode!=null){
                    zjkReviewProductVo.setReviewSn(byCode.getDesc());
                }
            }
        }
        return result;
    }

    @Override
    public TableDataInfo<ZjkReviewProductVo> getReviewProductListByExpert(ZjkReviewProductBo bo, PageQuery pageQuery) {
        Page<ZjkReviewProductVo> result = baseMapper.getReviewProductListByExpert(pageQuery.build(), bo);
        return TableDataInfo.build(result);
    }

    @Override
    public Long getReviewCount() {
        LambdaQueryWrapper<ZjkReviewPhase> eq = Wrappers.lambdaQuery();
        eq.eq(ZjkReviewPhase::getEffective, 1);
        return baseMapper.selectCount(eq);
    }

    @Override
    public Long getReviewProductCountByExpertId(Long expertId) {
        return baseMapper.getReviewProductCountByExpertId(expertId);
    }

    @Override
    public R<List<ZjkProductBo>> invoiceProductList() {
        List<ZjkProductBo> res = baseMapper.getInvoiceProductList();
        return R.ok(res);
    }

    private List<String> getReviewSn(String reviewSn) {
        List<String> list = new ArrayList<>();
        if(reviewSn.equals(ReviewSnEnum.FINAL_ACCEPTANCE.getCode())){
            list.add(ReviewSnEnum.FINAL_ACCEPTANCE.getCode());
            list.add(ReviewSnEnum.PRELIMINARY_ACCEPTANCE.getCode());
            list.add(ReviewSnEnum.CALL_BIDS.getDesc());
            list.add(ReviewSnEnum.PROJECT_APPROVAL.getDesc());
        }else if(reviewSn.equals(ReviewSnEnum.PRELIMINARY_ACCEPTANCE.getCode())){
            list.add(ReviewSnEnum.PRELIMINARY_ACCEPTANCE.getCode());
            list.add(ReviewSnEnum.CALL_BIDS.getDesc());
            list.add(ReviewSnEnum.PROJECT_APPROVAL.getDesc());
        }else if(reviewSn.equals(ReviewSnEnum.CALL_BIDS.getCode())){
            list.add(ReviewSnEnum.CALL_BIDS.getDesc());
            list.add(ReviewSnEnum.PROJECT_APPROVAL.getDesc());
        }else if(reviewSn.equals(ReviewSnEnum.PROJECT_APPROVAL.getCode())){
            list.add(ReviewSnEnum.PROJECT_APPROVAL.getCode());
        }
        return list;
    }
}
